/*****************************************************************************
 * Project: RooFit                                                           *
 *                                                                           *
 * Copyright (c) 2000-2005, Regents of the University of California          *
 *                          and Stanford University. All rights reserved.    *
 *                                                                           *
 * Redistribution and use in source and binary forms,                        *
 * with or without modification, are permitted according to the terms        *
 * listed in LICENSE (http://roofit.sourceforge.net/license.txt)             *
 *****************************************************************************/

#ifndef ROOABSCACHEDPDF
#define ROOABSCACHEDPDF

#include "RooAbsPdf.h"
#include "RooRealProxy.h"
#include "RooAbsReal.h"
#include "RooHistPdf.h"
#include "RooObjCacheManager.h"
#include "RooAICRegistry.h"
#include "RooChangeTracker.h"

#include <map>

class RooAbsCachedPdf : public RooAbsPdf {
public:

  // Default constructor
  RooAbsCachedPdf() : _cacheMgr(this,10) {}
  RooAbsCachedPdf(const char *name, const char *title, int ipOrder=0);
  RooAbsCachedPdf(const RooAbsCachedPdf& other, const char* name=nullptr) ;

  double getValV(const RooArgSet* set=nullptr) const override ;
  bool selfNormalized() const override {
    // Declare p.d.f self normalized
    return true ;
  }

  RooAbsPdf* getCachePdf(const RooArgSet& nset) const {
    // Return RooHistPdf that represents cache histogram
    return getCachePdf(&nset) ;
  }
  RooDataHist* getCacheHist(const RooArgSet& nset) const {
    // Return RooDataHist with cached values
    return getCacheHist(&nset) ;
  }
  RooAbsPdf* getCachePdf(const RooArgSet* nset=nullptr) const ;
  RooDataHist* getCacheHist(const RooArgSet* nset=nullptr) const ;

  void setInterpolationOrder(int order) ;
  Int_t getInterpolationOrder() const {
    // Set interpolation order in RooHistPdf that represent cached histogram
    return _ipOrder ;
  }

  bool forceAnalyticalInt(const RooAbsArg& dep) const override ;
  Int_t getAnalyticalIntegralWN(RooArgSet& allVars, RooArgSet& analVars, const RooArgSet* normSet, const char* rangeName=nullptr) const override ;
  double analyticalIntegralWN(Int_t code, const RooArgSet* normSet, const char* rangeName=nullptr) const override ;

  std::unique_ptr<RooAbsArg> compileForNormSet(RooArgSet const &normSet, RooFit::Detail::CompileContext & ctx) const override;

  class PdfCacheElem : public RooAbsCacheElement {
  public:
    PdfCacheElem(const RooAbsCachedPdf& self, const RooArgSet* nset) ;

    // Cache management functions
    RooArgList containedArgs(Action) override ;
    void printCompactTreeHook(std::ostream&, const char *, Int_t, Int_t) override ;

    RooHistPdf* pdf() { return _pdf.get() ; }
    RooDataHist* hist() { return _hist.get() ; }
    const RooArgSet& nset() { return _nset ; }
    RooChangeTracker* paramTracker() { return _paramTracker.get() ; }

  private:
    // Payload
    std::unique_ptr<RooHistPdf>  _pdf ;
    std::unique_ptr<RooChangeTracker> _paramTracker ;
    std::unique_ptr<RooDataHist> _hist ;
    RooArgSet    _nset ;
    std::unique_ptr<RooAbsReal>  _norm ;

  } ;

  protected:

  void computeBatch(cudaStream_t*, double* output, size_t size, RooFit::Detail::DataMap const&) const override;

  PdfCacheElem* getCache(const RooArgSet* nset, bool recalculate=true) const ;

  virtual const char* payloadUniqueSuffix() const { return nullptr ; }

  friend class PdfCacheElem ;
  virtual const char* binningName() const {
    // Return name of binning to be used for creation of cache histogram
    return "cache" ;
  }
  virtual PdfCacheElem* createCache(const RooArgSet* nset) const {
    // Create cache storage element
    return new PdfCacheElem(*this,nset) ;
  }
  virtual const char* inputBaseName() const = 0 ;
  virtual RooFit::OwningPtr<RooArgSet> actualObservables(const RooArgSet& nset) const = 0 ;
  virtual RooFit::OwningPtr<RooArgSet> actualParameters(const RooArgSet& nset) const = 0 ;
  virtual RooAbsArg& pdfObservable(RooAbsArg& histObservable) const { return histObservable ; }
  virtual void fillCacheObject(PdfCacheElem& cache) const = 0 ;

  mutable RooObjCacheManager _cacheMgr ; //! The cache manager
  Int_t _ipOrder ; // Interpolation order for cache histograms

  std::string cacheNameSuffix(const RooArgSet& nset) const ;
  virtual TString histNameSuffix() const { return TString("") ; }
  void disableCache(bool flag) {
    // Flag to disable caching mechanism
    _disableCache = flag ;
  }

  mutable RooAICRegistry _anaReg ; ///<! Registry for analytical integration codes
  class AnaIntConfig {
  public:
    RooArgSet _allVars ;
    RooArgSet _anaVars ;
    const RooArgSet* _nset ;
    bool    _unitNorm ;
  } ;
  mutable std::map<Int_t,AnaIntConfig> _anaIntMap ; ///<! Map for analytical integration codes



private:

  bool _disableCache = false; ///< Flag to run object in passthrough (= non-caching mode)

  ClassDefOverride(RooAbsCachedPdf,2) // Abstract base class for cached p.d.f.s
};

#endif
