// $Id: matrixRoot.cxx,v 1.25 2005/06/06 08:04:38 roberfro Exp $
// Class matrix : to create a new matrix from a file
// Author: V.Roberfroid <mailto:roberfroid@fynu.ucl.ac.be>
#include "matrixRoot.h"
using namespace std;

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// matrixRoot                                                           //
// ----------                                                           //
//                                                                      //
// matrixRoot is a class to manage ascii files containing numbers.      //
// The default constructor create a TMatrix with these data.            //
// Begin_Html                                                           //
// Here is some examples of methods :                                   //
// <LI> add or delete a row or a colomn.                                //
// <LI> draw a colomn as a function of an other using TGraph or         //
// TGraphErrors.                                                        //
// <LI> add, substract, multiply or divide term by term the matrixRoot  //
// <LI> sort the matrixRoot from a specific column                      //
// <LI> select a part of the matrixRoot by you own selections           //
// <LI> transform the matrixRoot in TNtuple.                            //
// End_Html                                                             //
// Example of using matrixRoot : Begin_Html <A HREF="examples/matrixRoot.C.html"> matrixRoot.C </A> End_Html //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

 ClassImp(matrixRoot)

//_____________________________________________________________________
matrixRoot::matrixRoot()
{
	NRow=0; NCol = 0; Nelt=0;
}
//_____________________________________________________________________
matrixRoot::~matrixRoot()
{
}
//_____________________________________________________________________
matrixRoot::matrixRoot(char *filename)
{
// Constructor : read filename to create a new matrixRoot
  Fill(filename);
}
//_____________________________________________________________________
matrixRoot::matrixRoot(int nr,int nc)
{
	NRow = nr; NCol = nc; Nelt = NRow*NCol;
	if(nr != 0 && nc != 0) fIndex.ResizeTo(NRow,NCol);
}
//_____________________________________________________________________
inline matrixRoot::matrixRoot(const matrixRoot &m)
{
//  Copy constructor
      this->NRow=m.NRow;
      this->NCol=m.NCol;
      this->Nelt=m.Nelt;
      if(!IsEmpty())
      {
	fIndex.ResizeTo(NRow, NCol);
	for(int nr=0; nr < m.NRow; nr++)
	  for(int nc=0; nc < m.NCol; nc++)
		fIndex(nr,nc)=m.fIndex(nr,nc);
      }
}
//_____________________________________________________________________
matrixRoot::matrixRoot(const TVector &v)
{
        NRow = v.GetNoElements();
        NCol = 1;
        Nelt = NRow;
        if(NRow > 0)
        {
          fIndex.ResizeTo(NRow, NCol);
          for(int nr=0; nr < NRow; nr++)
                fIndex(nr,0)=v[nr];
        }
}
//_____________________________________________________________________
int matrixRoot::IsEmpty()
{
	if(NRow == 0 || NCol == 0) 
	{
		cout << "Error in matrixRoot: Matrix empty" << endl;
		return 1;
	}
	else return 0;
}
//_____________________________________________________________________
void matrixRoot::Fill(char *filename)
{
// Fill the matrixRoot from the file "filename"
  NRow=0; NCol=0; Nelt=0;
  ifstream infile(filename);
  if(!infile)
    {
      cout << "Error in matrixRoot::Fill: Cannot open " << filename << endl;
      return;
    }
  double a; char c; int nc=0, nr=0;
  vector<double> vect;

  while(!infile.eof())
    {
      char s[1024] = {'\0'}; int i=0;
      infile.get(c);
      if(c == '#') {
         do {infile.get(c);}
         while (c != '\n');
      }
      else 
      {
        // To read just digit
        for(i=0; (c >= '0' && c <= '9') || (c == '.') || (c == '-') || (c == 'e') || (c == 'E'); i++)
	{
	  s[i]=c;
	  infile.get(c);
	}
        if(i != 0) { vect.push_back(atof(s)); Nelt++; nc++; }
        if(c=='\n') { 
           nr++; 
           if(NCol==0) NCol = nc;
           else if(nc != NCol && nc!=0) { 
               cout << "Error in matrixRoot::Fill: the number of column is not the same than the previous one (line:";
               cout << nr << ")" << endl;
               infile.close();
               return;
           }
           nc=0;
        }
      }
    }
  infile.close();
  NRow = nr-1;
  if(NRow <= 0 || Nelt <= 0) 
    {cout << "Error in matrixRoot::Fill: Problems in the input file" << endl; return;}
  NCol = Nelt/NRow;
  fIndex.ResizeTo(NRow, NCol);
  for(nr=0; nr < NRow; nr++) 
    for(nc=0; nc < NCol; nc++) fIndex(nr,nc) = vect[NCol*nr + nc];

}
//-------------------------------------------------------------------
void matrixRoot::Print()
{
   //  Print the matrixRoot in the TMatrix format
  if(!IsEmpty()) fIndex.Print();
}
//-------------------------------------------------------------------
void matrixRoot::PrintAscii()
{
   //  Print the matrixRoot in the ascii format
	int nc, nr; 
	if(!IsEmpty())
	{
	   for(nr=0; nr < NRow; nr++)
	   {
		for(nc=0; nc < NCol; nc++)
			cout << fIndex(nr,nc) << "  ";
		cout << endl;
	   }
	}
}
//-------------------------------------------------------------------
void matrixRoot::PrintAscii(char *filename)
{
   //  Print the matrixRoot, in the ascii format, in a file
	FILE *fp;
	fp = fopen (filename, "w");
	int nc, nr; 
	if(!IsEmpty())
	{
	   for(nr=0; nr < NRow; nr++)
	   {
		for(nc=0; nc < NCol; nc++)
			fprintf(fp,"%f  ",fIndex(nr,nc));
		fprintf(fp,"\n");
	   }
	}
}
//-------------------------------------------------------------------
TVector  matrixRoot::GetRow(int numrow)
{
// To get the row numrow (TVector)
  TVector line(NCol);
  if(numrow < NRow) 
  { 
   for(int i=0; i<NCol; i++) line(i) = fIndex(numrow,i);
  }
  else cout << "Error in matrixRoot::GetRow: numrow > NRow (in matrixRoot)" << endl;  
  return line;
}
//-------------------------------------------------------------------
TVector  matrixRoot::GetRow(int numrow, int collw, int colup)
{
// To get a part of the row numrow between collw and colup
  TVector line(colup-collw);
  if(numrow < NRow && colup < NCol && collw < NCol) 
    for(int i=collw; i<colup; i++) line(i-collw) = fIndex(numrow,i);
  else
    cout << "Error in matrixRoot::GetRow: numrow > NRow || colup > NCol || collw > NCol" << endl; 
  return line;
}
//-------------------------------------------------------------------
TVector  matrixRoot::GetCol(int numcol)
{
// To get the colomn numcol
  TVector col(NRow);
  if(numcol < NCol) 
    for(int i=0; i<NRow; i++) col(i) = fIndex(i,numcol);
  else cout << "Error in matrixRoot::GetCol : numcol > NCol" << endl;
  return col;
}
//_____________________________________________________________________
TVector  matrixRoot::GetCol(int numcol, int rowlw, int rowup)
{
// To get a part of the colomn numcol between rowlw and rowup
   TVector col(rowup-rowlw + 1);
  if(numcol < NCol && rowup < NRow && rowlw < NRow)
   for(int i=rowlw; i<=rowup; i++) col(i-rowlw) = fIndex(i,numcol); 
  else
	  cout << "Error in matrixRoot::GetCol : numcol > NCol || rowup > NRow || rowlw > NRow" << endl;
   return col;
}
//_____________________________________________________________________
double   matrixRoot::GetSumCol(int ncol)
{
// return the sum of all the components of the column with index ncol
	double sum = 0;
	for(int i=0; i<NRow; i++) sum+=fIndex(i,ncol);
	return sum;
}
//_____________________________________________________________________
void   matrixRoot::AddRow(TVector v)
{
// To add a new row at the end of the matrix
	int nr, nc;
	NCol = (NCol == 0) ? v.GetNoElements() : NCol;
	if(v.GetNoElements() != NCol) 
	{
          cout << "Error in matrixRoot::AddRow: TVector::GetNoElements() != NCol" 
               << endl;
	  return;
	}
	NRow=NRow+1;
	if(NRow > 1)
	{
	  TMatrixD a(fIndex);
	  fIndex.ResizeTo(NRow,NCol);
	  for(nr=0; nr<NRow-1; nr++)
	 	for(nc=0;nc<NCol; nc++) fIndex(nr,nc)=a(nr,nc);
	}
	if(NRow == 1) fIndex.ResizeTo(NRow,NCol);
	for(nc=0; nc < NCol; nc++) fIndex(NRow-1,nc) = v(nc);
}
//_____________________________________________________________________
void   matrixRoot::AddCol(TVector v)
{
// To add a new colomn at the end of the matrix
	int nr, nc;
	NRow = (NRow == 0) ? v.GetNoElements() : NRow;
	if(v.GetNoElements() != NRow) 
	{
	  cout << "Error in matrixRoot::AddCol: TVector::GetNoElements() != NRow (in matrixRoot)" 
               << endl;
	  return;
	}
	NCol=NCol+1;
	if(NCol > 1)
	{
	  TMatrixD a(fIndex);
	  fIndex.ResizeTo(NRow,NCol);
	  for(nr=0; nr<NRow; nr++)
		for(nc=0;nc<NCol-1; nc++) fIndex(nr,nc)=a(nr,nc);
	}
	if(NCol == 1) fIndex.ResizeTo(NRow,NCol);
	for(nr=0; nr < NRow; nr++) fIndex(nr,NCol-1) = v(nr);
}
//_____________________________________________________________________
void matrixRoot::Delete()
{
	delete this;
}
//_____________________________________________________________________
void matrixRoot::DeleteRow(int nrow)
{
// To delete the row nrow
     if(!IsEmpty())
     {
	if(nrow > NRow) 
	{
		cout << "Error in matrixRoot::DeleteRow: nr > NRow (in matrixRoot)"
		     << endl;
		return;
	}
	int nc, nr;
	matrixRoot a(*this);
	NRow=1;
	fIndex.ResizeTo(NRow,NCol);
	int firstrow = (nrow==0) ? 1 : 0;
	for(nc=0; nc < a.GetNCol(); nc++) fIndex(0,nc) = a(firstrow,nc);
	for(nr=firstrow+1; (nr<a.GetNRow()); nr++)
		if(nr != nrow) AddRow(a.GetRow(nr));
     }
}
//_____________________________________________________________________
void matrixRoot::DeleteCol(int ncol)
{
// To delete the row nrow
     if(!IsEmpty())
     {
	if(ncol > NCol) 
	{
		cout << "Error in matrixRoot::DeleteCol: ncol > NCol (in matrixRoot)"
		     << endl;
		return;
	}
	int nc, nr;
	matrixRoot a(*this);
	NCol=1;
	fIndex.ResizeTo(NRow,NCol);
	int firstcol = (ncol==0) ? 1 : 0;
	for(nr=0; nr < a.GetNRow(); nr++) fIndex(nr,0) = a(nr,firstcol);
	for(nc=firstcol+1; (nc<a.GetNCol()); nc++)
		if(nc != ncol) AddCol(a.GetCol(nc));
     }
}
//_____________________________________________________________________
inline
double& matrixRoot::operator() (unsigned row, unsigned col)
{
// Operator () to get the value of matrixRoot(row,col)
    if (row >= NRow && col >= NCol) 
	    cout << "Error in matrixRoot::operator(): Bad index" << endl; 
    return fIndex(row,col);
}

inline
double matrixRoot::operator() (unsigned row, unsigned col) const
{
// Operator () to get the value of matrixRoot(row,col)
    if (row >= NRow && col >= NCol) 
	    cout << "Error in matrixRoot::operator(): Bad index" << endl; 
    return fIndex(row,col);
}
//_____________________________________________________________________
matrixRoot& matrixRoot::operator=(const matrixRoot& a)
{
	NRow = a.NRow;
	NCol = a.NCol;
	Nelt = a.Nelt;
	if(a.NRow > 0 && a.NCol > 0)
	{
	  fIndex.ResizeTo(NRow, NCol);
	  for(int nr=0; nr < a.NRow; nr++)
	    for(int nc=0; nc < a.NCol; nc++)
		fIndex(nr,nc)=a.fIndex(nr,nc);
	}
	return *this;
}
//_____________________________________________________________________
matrixRoot& matrixRoot::operator=(const TVector& v)
{
        NRow = v.GetNoElements();
        NCol = 1;
        Nelt = NRow;
        if(NRow > 0)
        {
          fIndex.ResizeTo(NRow, NCol);
          for(int nr=0; nr < NRow; nr++)
                fIndex(nr,0)=v[nr];
        }
        return *this;
}
//_____________________________________________________________________
void  matrixRoot::SortByCol(int ncol, int nrowmin, int nrowmax)
{
     // Sort the row of the matrixRoot as a function of the colomn ncol 
     // between nrowmin and nrowmax included.
  int gap, i, j , k;
  double temp;
  int n = nrowmax-nrowmin + 1;
  if(!IsEmpty())
  {
    if(nrowmax >= NRow  || ncol >= NCol ) 
    {
	  cout << "Error in matrixRoot::SortByCol : nrowmax >= NRow  || ncol >= NCol " 
	       << endl;
	  return;
    }
   for(gap = n/2; gap > 0; gap /=2)
     for(i=gap;i<n; i++)
      for(j=i-gap; 
	  j>=0 && fIndex(j+nrowmin,ncol)>fIndex(j+nrowmin+gap,ncol); 
	  j-=gap) {
	for(k=0; k < NCol; k++){
	  temp = fIndex(j+nrowmin,k);
	  fIndex(j+nrowmin,k) = fIndex(j+nrowmin+gap,k);
	  fIndex(j+nrowmin+gap,k)=temp;
	}
      }
  }
}
//_____________________________________________________________________
void  matrixRoot::SortByCol(int ncol) 
{ 
     // idem than SortByCol(int ncol, int nrowmin, int nrowmax) 
     // but for the complete matrixRoot.
  this->SortByCol(ncol,0,this->GetNRow()-1);
}
//_____________________________________________________________________
TGraph*  matrixRoot::GetTGraph(int ncolX, int ncolY)
{
	// To Get a TGraph : the colomn ncolY as a function af ncolY
	// for all the rows
  TGraph *gr;
  if(!IsEmpty())
  {
    if(ncolX < NCol && ncolY < NCol) 
    {
        TVector coloX(GetNRow());
        coloX = GetCol(ncolX);         

        TVector coloY(GetNRow());
        coloY = GetCol(ncolY);         // colo est la colonne 0
        gr = new TGraph(coloX,coloY);
    }
    else
	  cout << "Error in matrixRoot::GetTGraph : ncolX >= NCol || ncolY >= NCol" 
	       << endl;
  }
  return gr;
}
//_____________________________________________________________________
TGraph*  matrixRoot::GetTGraph(int ncolX, int ncolY, int nrowmin, int nrowmax)
{
	// To Get a TGraph : the colomn ncolY as a function af ncolY
	// for a part of the rows (between nrowmin and nrowmax)
  TGraph *gr;
  if(!IsEmpty())
  {
   if(ncolX < NCol && ncolY < NCol)
   {
      TVector coloX(nrowmax - nrowmin  + 1 );
      coloX = GetCol(ncolX,nrowmin,nrowmax);         

      TVector coloY(nrowmax - nrowmin + 1 );
      coloY = GetCol(ncolY,nrowmin,nrowmax);         // colo est la colonne 0
      gr = new TGraph(coloX,coloY);
    }
    else
        cout << "Error in matrixRoot::GetTGraph : ncolX >= NCol || ncolY >= NCol" 
	     << endl;
  }
  return gr;
}
//_____________________________________________________________________
TGraphErrors* matrixRoot::GetTGraphErrors(int ncolX, int ncolY, 
		                          int ncolDX, int ncolDY)
{
	// idem than TGraph but for TGraphErrors
	// ncolDX is the number of the colomn containing the errors on X values
	// ncolDY is the number of the colomn containing the errors on Y values
  TGraphErrors *gr;
  if(!IsEmpty())
  {
   if(ncolX < NCol && ncolY < NCol && ncolDX < NCol && ncolY < NCol)
    {
       double x[NRow], y[NRow], Dx[NRow], Dy[NRow];
       int i;
  
       for(i=0; i < NRow; i++) 
       {
	  x[i] = fIndex(i,ncolX);    y[i] = fIndex(i,ncolY);
	  Dx[i] = fIndex(i,ncolDX);  Dy[i] = fIndex(i,ncolDY);
       }
       gr = new TGraphErrors(NRow,x,y,Dx,Dy);
    }
   else
        cout << "Error in matrixRoot::GetTGraphErrors : ncolX >= NCol || ncolY >= NCol" 
	     << endl;
  }
  return gr;
}
//_____________________________________________________________________
TGraphErrors* matrixRoot::GetTGraphErrors(int ncolX, int ncolY, 
		                          int ncolDY)
{
	// idem than TGraph but for TGraphErrors
	// ncolDY is the number of the colomn containing the errors on Y values
	// there is no errors on X
  TGraphErrors *gr;
  if(!IsEmpty())
  {
   if(ncolX < NCol && ncolY < NCol && ncolDY < NCol)
    {
      double x[NRow], y[NRow], Dy[NRow];
      int i;

  
      for(i=0; i < NRow; i++) 
      {
	  x[i] = fIndex(i,ncolX);    y[i] = fIndex(i,ncolY);
	  Dy[i] = fIndex(i,ncolDY);  
      }
      gr = new TGraphErrors(NRow,x,y,0,Dy);
    }
   else
        cout << "Error in matrixRoot::GetTGraphErrors : ncolX >= NCol || ncolY >= NCol"
             << endl;
  }
   return gr;
}
//_____________________________________________________________________
TNtuple* matrixRoot::GetTNtuple(char *NtupleName)
{
// To Get a Ntuple from the matrixRoot. The names of the different variables are :
// c0, c1, c2, c3, ... 
// To draw : ntuple->Draw("c1:c3") for example.  
	TString s;
	for(int ncol=0; ncol < NCol; ncol++) { s += "c"; s += ncol; s+=":";}
	cout << "Variables " << s <<  "  were created" << endl;
	TNtuple *ntFromTMatrix = new TNtuple(NtupleName,
			"Ntuple from a TMatrix",s);
	float row[NCol];
	for(int nrow = 0; nrow < NRow; nrow++)
	{	
		for(int ncol=0; ncol < NCol; ncol++) row[ncol] = (float)fIndex(nrow,ncol);
		ntFromTMatrix->Fill(row);
	}
	return ntFromTMatrix;
}
//_____________________________________________________________________
matrixRoot matrixRoot::Select(int ncol,char *condition)
{
// To select a part of the matrix. 
// Exemple : b=a.Select(3,"6 < x && x < 12")
// b contains only element of a where the third colomn 
// satisfied the condition "6 < x && x < 12"
     matrixRoot c;
     if(!IsEmpty())
     {
	int nr;
	TFormula Test("Test",condition);
	matrixRoot b(1,NCol);  
	for(nr=0; nr < NRow; nr++) if(Test.Eval(fIndex(nr,ncol))) 
			b.AddRow(GetRow(nr));
	if(b.GetNRow() != 1) 
	{
		b.DeleteRow(0);
		return b;
	}
	else 
	{
		cout << "Error in matrixRoot::Select : No element in the colomn " << ncol << " where "
	             << condition << endl;
		return c;
	}
     }
     else return c;
}	
//_____________________________________________________________________
matrixRoot matrixRoot::Execute(TString formula)
{
	// Execute a formula on all components in the matrix
	// Example : matrixRoot b = a.Execute("sqrt(x)");
	// In this example b contains the square root components of a. 
        TFormula f("f",formula);
	matrixRoot b(NRow,NCol);
	for(int nr=0; nr < NRow; nr++)
	    for(int nc=0; nc < NCol; nc++)
		    b(nr,nc)=f.Eval(fIndex(nr,nc));
	return b;
}
//_____________________________________________________________________
matrixRoot matrixRoot::operator+(const matrixRoot& a)
{
	matrixRoot c(a.NRow,a.NCol);
	if(!IsEmpty())
	{
	  if(a.NRow == NRow && a.NCol == NCol)
	  {
	
  	   for(int nr=0; nr < a.NRow; nr++)
	       for(int nc=0; nc < a.NCol; nc++)
		       c(nr,nc)=a(nr,nc)+fIndex(nr,nc);
	  }
	  else
	    cout << "Error in matrixRoot::operator+ : Addition of two matrixRoot with different dimensions"
	         << endl;
	}	  
	return c;
}
//_____________________________________________________________________
matrixRoot matrixRoot::operator+(const double& a)
{
	matrixRoot c(NRow,NCol);

	for(int nr=0; nr < NRow; nr++)
		for(int nc=0; nc < NCol; nc++)
			c(nr,nc)=a+fIndex(nr,nc);
	return c;
}
//_____________________________________________________________________
/*
matrixRoot operator+(matrixRoot a, double b)
{
	matrixRoot c(a.GetNRow(),a.GetNCol());
	c = b+a; 
	return c ;
}
*/
//_____________________________________________________________________
matrixRoot matrixRoot::operator-(const matrixRoot& a)
{
      matrixRoot c(a.NRow,a.NCol);
      if(a.NRow == NRow && a.NCol == NCol)
      {
	
	for(int nr=0; nr < a.NRow; nr++)
	       for(int nc=0; nc < a.NCol; nc++)
		       c(nr,nc)=a(nr,nc)-fIndex(nr,nc);
      }
      else
      {
           cout << "Error in matrixRoot::operator-: Addition of two matrixRoot with different"; 
           cout <<   " dimensions " << endl;
      }
      return c;
}
//_____________________________________________________________________
/*
matrixRoot matrixRoot::operator-(const double& a)
{
	matrixRoot c(NRow,NCol);
	c=a+(-1.*this);
	return c;
}
*/
//_____________________________________________________________________
/*
matrixRoot operator-(matrixRoot a, double b)
{
	matrixRoot c(a.GetNRow(),a.GetNCol());
	c=(-1.)*(b-a);
	return  c;
}
*/
//_____________________________________________________________________
matrixRoot matrixRoot::operator*(const double& a)
{
        matrixRoot c(NRow,NCol);

        for(int nr=0; nr < NRow; nr++)
               for(int nc=0; nc < NCol; nc++)
                       c(nr,nc)=a*fIndex(nr,nc);
        return c;
}
//_____________________________________________________________________
matrixRoot matrixRoot::operator*(const matrixRoot& a)
// multiply term by term the 2 matrix
{
      matrixRoot c(NRow,NCol);

      if(a.NRow == NRow && a.NCol == NCol)
      {
        for(int nr=0; nr < NRow; nr++)
               for(int nc=0; nc < NCol; nc++)
                       c(nr,nc)=fIndex(nr,nc)*a(nr,nc);
      }
      else
      {
           cout << "Error in matrixRoot::operator*: Division of two matrixRoot with different";
           cout <<   " dimensions " << endl;
      }
      return c;
}
//_____________________________________________________________________
/*
matrixRoot operator*(matrixRoot b, double a)
{
	matrixRoot c(b.GetNRow(),b.GetNCol());
	c=a*b;
        return c;
}
*/
//_____________________________________________________________________
matrixRoot matrixRoot::operator/(const double& a)
{
        matrixRoot c(NRow,NCol);

        for(int nr=0; nr < NRow; nr++)
               for(int nc=0; nc < NCol; nc++)
                    if(fIndex(nr,nc) != 0)
                       c(nr,nc)=a/fIndex(nr,nc);
                    else
                      {
                       cout << "Error in matrixRoot::operator/: Divide by zero !"; 
		       cout << "(nr = " << nr << ", nc = " << nc << ")" << endl;
                      }
        return c;
}
//_____________________________________________________________________
matrixRoot matrixRoot::operator/(const matrixRoot& a)
{
      matrixRoot c(NRow,NCol);

      if(a.NRow == NRow && a.NCol == NCol)
      {
        for(int nr=0; nr < NRow; nr++)
               for(int nc=0; nc < NCol; nc++)
                    if(a(nr,nc) != 0)
                       c(nr,nc)=fIndex(nr,nc)/a(nr,nc);
                    else
                      {
                       cout << "Error in matrixRoot::operator/: Divide by zero !";
                       cout << "(nr = " << nr << ", nc = " << nc << ")" << endl;
                      }
      }
      else
      {
           cout << "Error in matrixRoot::operator/: Division of two matrixRoot with different";
           cout <<   " dimensions" << endl;
      }
      return c;
}
//_____________________________________________________________________
/*
matrixRoot operator/(matrixRoot a, double b)
{
     matrixRoot c(a.GetNRow(),a.GetNCol());
     if(b != 0)
     {
	     c=a*(1/b);
             return c;
     }
     else 
       {
         cout << "Divide by zero !" << endl;
         exit(-1);
       }
}
*/
//_____________________________________________________________________
